﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Graphics;
using Microsoft.Xna.Framework.Content;
using CashierCrisis.Entities;
using CashierCrisis.Managers;
using CashierCrisis.Behaviors;
using CashierCrisis.Graphs;
 
namespace CashierCrisis
{
    public class Stage
    {
        public EntityManager entityManager;
        public Entity Player { get; set; }
        public Entity Register { get; set; }
        public Vector2 DoorPosition { get; set; }
        public List<Entity> Customers { get; set; }
        public List<Entity> ItemShelves { get; set; }
        private EntityPool pool = new EntityPool(200);
        private Queue<CustomerData> customerSpawnQueue = new Queue<CustomerData>();
        private float interval = 6400;
        private float elapsed;
        private float total = 0;
        private int preloadedCustomers = 64;
        private int numberOfCustomers = 10;
        List<CustomerData> dataList = new List<CustomerData>();
        Queue<Vector2> customerQ = new Queue<Vector2>();


        public Stage()
        {
            entityManager = CashierGame.Instance.Services.GetService(typeof(EntityManager)) as EntityManager;
            Customers = new List<Entity>();
        }

        public void Initialize(ContentManager content)
        {

            // Door
            DoorPosition = new Vector2(64);

            //Cash Register & Counter
            Register = CreateRegister(content);

            //Create Shelves
            ItemShelves = CreateShelves(4, content);

            //Invalidate nodes that collide with item shelves.
            for (int i = 0; i < ItemShelves.Count; i++)
            {
                StaticSpriteComponent sprite = ItemShelves[i].GetComponent("sprite") as StaticSpriteComponent;
                Rectangle rect = new Rectangle(
                    (int)ItemShelves[i].Position.X,
                    (int)ItemShelves[i].Position.Y,
                    sprite.Sprite.Width,
                    sprite.Sprite.Height);

                for (int j = 0; j < PathPlanner.Graph.Nodes.Count; j++)
                {
                    if (rect.Contains(new Point((int)PathPlanner.Graph.Nodes[j].Position.X, (int)PathPlanner.Graph.Nodes[j].Position.Y)))
                    {
                        PathPlanner.Graph.Nodes[j].Index = GraphNode.InvalidNodeIndex;
                    }
                }
            }

            //Create Player
            Player = CreatePlayer(content);

            //Create Customers
            CreateCustomers(content);


        
        }



        public void Update(GameTime gameTime)
        {
            elapsed = (float)gameTime.ElapsedGameTime.Milliseconds;

            // Spawning Customers
            total += elapsed;

            if (total >= interval)
            {
                if (customerSpawnQueue.Count > 0)
                {
                    CustomerData c = customerSpawnQueue.Dequeue();
                    SpawnCustomer(c);
                    total = 0f;
                }
            }

            // Mood Growth when player is in radius.
            MoodGrowth(CustomersInRadius(Player,true), true);

            // Mood Decay when player NOT in radius.
            MoodGrowth(CustomersInRadius(Player,false), false);
            
        }

        #region Helper Methods

        // Create Register
        public Entity CreateRegister(ContentManager content)
        {
            Entity e = new Entity();
            e.Position = new Vector2(8, 500);
            e.AddComponent(
                new StaticSpriteComponent("sprite")
                {
                    Sprite = content.Load<Texture2D>("Sprites//Stores//store1CounterWithRegister")
                });

            entityManager.AddEntity(e);

            customerQ.Enqueue(new Vector2(8, 150));
            customerQ.Enqueue(new Vector2(8, 200));
            customerQ.Enqueue(new Vector2(8, 250));
            customerQ.Enqueue(new Vector2(8, 300));
            customerQ.Enqueue(new Vector2(8, 350));
            return e;
        }

        public Vector2 FreeSpot()
        {
            Vector2 position = customerQ.Dequeue();
            customerQ.Enqueue(position);
            return position;
        }

        // Create Shelving
        public List<Entity> CreateShelves(int n, ContentManager content)
        {
            Texture2D texture = content.Load<Texture2D>("Sprites//Stores//shelf_large");
            List<Entity> entities = new List<Entity>();


                Entity shelf1 = new Entity();
                shelf1.AddComponent( new StaticSpriteComponent("sprite")
                    {
                        Sprite = texture
                    });

                shelf1.Position = new Vector2(210, 100);
                entityManager.AddEntity(shelf1);
                entities.Add(shelf1);

                Entity shelf2 = new Entity();
                shelf2.AddComponent(new StaticSpriteComponent("sprite")
                {
                    Sprite = texture
                });

                shelf2.Position = new Vector2(820, 100);
                entityManager.AddEntity(shelf2);
                entities.Add(shelf2);

                Entity shelf3 = new Entity();
                shelf3.AddComponent(new StaticSpriteComponent("sprite")
                {
                    Sprite = texture
                });

                shelf3.Position = new Vector2(210, 300);
                entityManager.AddEntity(shelf3);
                entities.Add(shelf3);

                Entity shelf4 = new Entity();
                shelf4.AddComponent(new StaticSpriteComponent("sprite")
                {
                    Sprite = texture
                });

                shelf4.Position = new Vector2(820, 300);
                entityManager.AddEntity(shelf4);
                entities.Add(shelf4);

              

            return entities;
        }


        //Create Player
        public Entity CreatePlayer(ContentManager content)
        {
            Entity e = new Entity();
            e.AddComponent(new InputComponent("input"));
            e.AddComponent(new StaticSpriteComponent("sprite") { Sprite = content.Load<Texture2D>("Sprites//Player//NewFranTest") });
            e.AddComponent(new BoundingComponent("bounding"));
            e.AddComponent(new CollisionComponent("collision") { Radius = 16384 });
            entityManager.AddEntity(e);

            return e;
        }


        // Create Customers
        public void CreateCustomers(ContentManager content)
        {
            Random random = new Random();
            List<Entity> temp = new List<Entity>();
            for (int i = 0; i < preloadedCustomers; i++)
            {
                Entity e = pool.Fetch();
                e.AddComponent(new StatusBarComponent("status")
                    {
                        Sprite = content.Load<Texture2D>("Sprites//moodbar")
                    });
                
                e.AddComponent(new MoodComponent("mood"));
                e.AddComponent(new PhysicsComponent("physics"));
                e.AddComponent(new SteeringComponent("steering"));
                e.AddComponent(
                    new CustomerSpriteComponent("sprites")
                    {
                        Color = (CustomerColor)random.Next(0,4)
                    });
                e.AddComponent(
                    new CollisionComponent("collision")
                    {
                        Radius = 128
                    });

                e.AddComponent(new ItemComponent("items"));
                
                BrainComponent brain = new BrainComponent("brain");
                e.AddComponent(brain);
                
                Behavior root = CreateBehavior(e);
                brain.Add(root);
             
                temp.Add(e);
            }

            for (int i = 0; i < preloadedCustomers; i++)
            {
                pool.Return(temp[i]);
            }

           
            // Test # of customers
            for (int i = 0; i < (numberOfCustomers/2); i++)
            {
                CustomerData data = GetRandomCustomerData();
                data.Name = "Man";
                dataList.Add(data);
            }

            for (int i = numberOfCustomers / 2; i < numberOfCustomers; i++)
            {
                CustomerData data = GetRandomCustomerData();
                data.Name = "Girl";
                dataList.Add(data);
            }

           
            for (int index = dataList.Count - 1; index > 0; index--)
            {
                int randomIndex = random.Next(index + 1);
                CustomerData a = dataList[randomIndex];
                CustomerData b = dataList[index];
                Swap(index, randomIndex);
                

            }

            for (int i = 0; i < dataList.Count; i++)
            {
                customerSpawnQueue.Enqueue(dataList[i]);
            }
            
        }

        public void Swap(int index, int randomIndex)
        {
            CustomerData x, y;
            x = dataList[index];
            y = dataList[randomIndex];

            dataList[randomIndex] = x;
            dataList[index] = y;
        }



        // Spawn Customer w/ Customer XML Data
        public void SpawnCustomer(CustomerData customerData)
        {
            Entity e = pool.Fetch();
            e.Position = DoorPosition;
            MoodComponent moodComponent = e.GetComponent("mood") as MoodComponent;
            moodComponent.Mood = customerData.Mood;
            moodComponent.MoodDecayRate = customerData.MoodDecayRate;
            moodComponent.MoodGrowthRate = customerData.MoodGrowthRate;
            CustomerSpriteComponent customerSpriteComponent = e.GetComponent("sprites") as CustomerSpriteComponent;
            customerSpriteComponent.CustomerName = customerData.Name;
            entityManager.AddEntity(e);
            Customers.Add(e);
        }

        // Mood Growth Method
        public void MoodGrowth(List<Entity> customers, bool growing)
        {
            for (int i = 0; i < customers.Count; i++)
            {
                MoodComponent m = customers[i].GetComponent("mood") as MoodComponent;
                m.IsGrowing = growing;
            }
        }

        // Default Behavior Tree Root
        public Behavior CreateBehavior(Entity entity)
        {
            InitializeBehavior init = new InitializeBehavior(entity);
            MersenneTwister random = new MersenneTwister((uint)entity.ID);

            Vector2 position = ItemShelves[random.Next(0,ItemShelves.Count)].Position;
            Log.WriteLine(LogCategory.Game, position.ToString());


            var root = new Selector();

            var move = new Move();
            move.Accept(init);
            move.Destination = Vector2.Zero;

            var par = new Parallel();

            var seq = new Sequence();
           
            ConditionCustomerMad con = new ConditionCustomerMad();
            con.IsContinuous = true;
            con.Accept(init);
            par.Add(con);
            par.Add(seq);
           
            var path = seq.Add(
                new FollowPath()
                {
                    Path = PathPlanner.CreatePathToPosition(Vector2.Zero, position )
                });
            path.Accept(init);

            var wait = seq.Add(
                new Wait()
                {
                    Miliseconds = random.Next(300,500)
                });
            if (random.Next(100) > 40)
            {
                MersenneTwister r = new MersenneTwister(328);
                Vector2 position2 = ItemShelves[r.Next(0, ItemShelves.Count)].Position;
                var path2 = seq.Add(
                     new FollowPath()
                     {
                         Path = PathPlanner.CreatePathToPosition(position, position2)
                     });

                path2.Accept(init);

                var path3 = seq.Add(
                    new FollowPath()
                    {
                        Path = PathPlanner.CreatePathToPosition(position2, FreeSpot())
                    });
                path3.Accept(init);
            }

            root.Add(par);
            root.Add(move);
            
            return root;
        }

        // Random Customer Data
        public CustomerData GetRandomCustomerData()
        {
            CustomerData c = new CustomerData();
            MersenneTwister random = new MersenneTwister();
            c.Name = (random.Next(1, 100) >= 50) ? "Man" : "Girl";
            c.Mood = random.Next(40, 80);
            c.MoodDecayRate = 2;
            c.MoodGrowthRate = 3;
            c.SameItemBonus = random.NextDouble() * random.Next(100);
            c.SimilarItemBonus = random.NextDouble() * random.Next(100);
            return c;
        }


        // Get customers in or out of customer radius.
        public List<Entity> CustomersInRadius(Entity entity, bool inRadius)
        {
            CollisionComponent c = entity.GetComponent("collision") as CollisionComponent;
            List<Entity> entities = new List<Entity>();

            for (int i = 0; i < Customers.Count; i++)
            {
                if (c.Collide(Customers[i]) == inRadius)
                {
                    entities.Add(Customers[i]);
                }

            
                
               
            }
            return entities;
        }

        #endregion

        // Gameplay Modification Methods.

        #region Gameplay Modding
        public void ToggleGrowth()
        {
            for (int i = 0; i < Customers.Count; i++)
            {
                MoodComponent m = Customers[i].GetComponent("mood") as MoodComponent;
                if (m.IsGrowing)
                {
                    m.IsGrowing = false;
                }
                else
                {
                    m.IsGrowing = true;
                }
            }
        }

        public void ToggleDecay()
        {
            for (int i = 0; i < Customers.Count; i++)
            {
                MoodComponent m = Customers[i].GetComponent("mood") as MoodComponent;
                if (m.IsDecaying)
                {
                    m.IsDecaying = false;
                }
                else
                {
                    m.IsDecaying = true;
                }
            }
        }


        public void IncreaseInterval(int amount)
        {
            interval += amount;
        }

        public void DecreaseInterval(int amount)
        {
            interval -= amount;
        }


        public void IncreaseMoodGrowthRate(double amount)
        {
            for (int i = 0; i < Customers.Count; i++)
            {
                MoodComponent moodComponent = Customers[i].GetComponent("mood") as MoodComponent;
                moodComponent.MoodGrowthRate += amount;
            }
        }

        public void DecreaseMoodGrowthRate(double amount)
        {
            for (int i = 0; i < Customers.Count; i++)
            {
                MoodComponent moodComponent = Customers[i].GetComponent("mood") as MoodComponent;
                moodComponent.MoodGrowthRate -= amount;
            }
        }

        public void IncreaseMoodDecayRate(double amount)
        {
            for (int i = 0; i < Customers.Count; i++)
            {
                MoodComponent moodComponent = Customers[i].GetComponent("mood") as MoodComponent;
                moodComponent.MoodDecayRate += amount;
            }
        }

        public void DecreaseMoodDecayRate(double amount)
        {
            for (int i = 0; i < Customers.Count; i++)
            {
                MoodComponent moodComponent = Customers[i].GetComponent("mood") as MoodComponent;
                moodComponent.MoodDecayRate -= amount;
            }
        }

        public void IncreaseSpeed(int amount)
        {
            for (int i = 0; i < Customers.Count; i++)
            {
                PhysicsComponent physicsComponent = Customers[i].GetComponent("physics") as PhysicsComponent;
                physicsComponent.MaxSpeed += amount;
            }
        }

        public void DecreaseSpeed(int amount)
        {
            for (int i = 0; i < Customers.Count; i++)
            {
                PhysicsComponent physicsComponent = Customers[i].GetComponent("physics") as PhysicsComponent;
                physicsComponent.MaxSpeed -= amount;
            }
        }

        #endregion

    }
}
